define(['app', 'angular'], function (app, angular) {
	"use strict";

	app.directive('notificationDeliveryOptions', function (PatientPreferenceService, formatter) {
		return {
			restrict: 'AE',
			require: 'ngModel',
			scope: {
				ngModel : '=',
				ngDisabled : '=',
				notifyMeLabel: '@',
				isRecurring : '=?',
				hideReminderTimes: '=?',
                hideDay: '=',
                hideStatusLabel: '=',
				nextDueDate: '=?',
				nextDueDateLabel: '@?',
				notificationDateOptionCount: '=?',
                enableStartDate: '=?'
			},
			templateUrl: 'modules/ui-components/form/controls/composite/notification-delivery-options/notification-delivery-options_template.html',
			link: function(scope, elem, attrs, ngModelCtrl) {
				scope.emailAddress = null;
				scope.isEmailAddressStored = null;
				scope.reminderTypes = [{type:"In-App"},{type:"Email"}];
				scope.notificationDates = getNotificationDateOptions();
				scope.notificationOptions = ["On", "Off"];
				scope.allowedDateRange = { min: '', max: '' };
				scope.allowedDateRange.min = formatter.getFormattedFrontendDate(new Date());
                scope.isStartDateEnabled = false;

				// for reminders in edit mode, reminders allow editing of fields other than startDate (which is disabled in edit mode)
				// so change allowedDateRange.min to prevent error stating the date must be today or later
				var unbindWatcher = scope.$watch('ngModel', function (newVal, oldVal) {
					if (scope.isRecurring && newVal && newVal.link) {
						scope.isReminderEventDateStored = !!scope.ngModel.eventDate;
						unbindWatcher();
					}
				}, true);

				function getNotificationDateOptions() {
					var dates = scope.isRecurring ? ['Daily', 'Weekly', 'Monthly'] : ['On the date', '1 day before', '2 weeks before', '4 weeks before'];
					return scope.notificationDateOptionCount && scope.notificationDateOptionCount < dates.length
						? dates.splice(0, scope.notificationDateOptionCount)
						: dates;
				}

				scope.nextDueDateLabel = scope.nextDueDateLabel ? scope.nextDueDateLabel : 'Next Due Date';
				scope.notifyMeErrorLabel = (scope.notifyMeLabel === "Remind Me to Create My Plan") ? scope.notifyMeLabel : "Notify Me";
				scope.notificationMessageType = scope.notifyMeLabel === 'Remind Me to Create My Plan' ? 'this appointment plan.' : 'the reminder.';
				scope.errorHandling = {
					'validNextDueDate': {
						message: 'Notifications cannot be provided for the ' + scope.nextDueDateLabel + ' you entered. Turn the Notification off to save ' + scope.notificationMessageType,
						priority: 1
					},
					'deliverySelected': {
						message: 'A Notification Delivery method is needed to save this record. Please select a method and save again.',
						priority: 2
					},
					'alertSelected': {
						message: 'The ' + scope.notifyMeErrorLabel + ' field is needed to save this record. Please select at least one option and save again.',
						priority: 2
					}
				};

				(function init(){
					PatientPreferenceService.getPreferencesData().then(function (response) {
						scope.emailAddress = response.emailAddress;
						scope.isEmailAddressStored = !!response.emailAddress;
					}, function () {
						scope.isEmailAddressStored = false;
					}).finally(function (){
						if (!scope.isEmailAddressStored){
							scope.ngModel.notificationTypes.forEach(function (value, index) {
								if (value === "Email"){
									scope.ngModel.notificationTypes.splice(index, 1);
								}
							});
						}
					});
				})();

				// ================ HELPERS ================
				var validArrayValues = function (arr) {
					var validArray = arr.some(function (element) {
						return !!element;
					});

					return (validArray && arr.length > 0);
				};

				var dateOptionMap = {
		    		'On the date' : 0,
		    		'1 day before' : 1,
		    		'2 weeks before' : 14,
		    		'4 weeks before' : 28
		    	};

				var getDayDifference = function (d1, d2) {
					var MS_PER_DAY = 1000 * 60 * 60 * 24,
					    utc1 = Date.UTC(d1.getFullYear(), d1.getMonth(), d1.getDate()),
					    utc2 = Date.UTC(d2.getFullYear(), d2.getMonth(), d2.getDate());

					return Math.floor((utc2 - utc1) / MS_PER_DAY);
				};

				var validateNextDueDate = function (next) {
                    var currentDate = new Date(),
                        nextDue = new Date(next),
                        dayDiff = getDayDifference(currentDate, nextDue),
                        validDate = dayDiff > 0 || scope.ngModel.notificationStatus === 'Off';

                    ngModelCtrl.$setValidity('validNextDueDate', !next || validDate);
				};

				// ================ WATCHES ================
				scope.$watch('[ngModel.alert, ngModel.notificationTypes, ngModel.notificationStatus]', function () {
					if (scope.ngModel && scope.ngModel.notificationStatus === 'On') {
						ngModelCtrl.$setValidity('deliverySelected', validArrayValues(scope.ngModel.notificationTypes));
						ngModelCtrl.$setValidity('alertSelected', scope.isRecurring || validArrayValues(scope.ngModel.alert));
						validateNextDueDate(scope.nextDueDate);
					} else {
						ngModelCtrl.$setValidity('alertSelected', true);
						ngModelCtrl.$setValidity('deliverySelected', true);
						ngModelCtrl.$setValidity('validNextDueDate', true);
					}
				}, true);

				scope.$watch('ngModel.alert', function (newVal, oldVal) {
					if (ngModelCtrl.$pristine && scope.ngModel && typeof oldVal !== 'undefined' && newVal !== oldVal) {
						ngModelCtrl.$setViewValue(ngModelCtrl.$viewValue);
					}
				}, true);

                scope.$watch('ngModel.notificationStatus', function (newVal, oldVal) {
                    if (scope.ngModel && newVal && oldVal && newVal !== oldVal) {
						if (newVal === 'On') {
							if(scope.enableStartDate) {
								scope.isStartDateEnabled = true;
							}
							scope.ngModel.notificationTypes = ["In-App"];
						} else {
							scope.ngModel.notificationTypes = [];
							scope.ngModel.alert = [];
						}
                    }
                }, true);

                scope.$watch('ngModel.eventDate', function (newVal, oldVal) {
                    if (scope.ngModel && newVal && newVal.length === 10) {
                        if (scope.isReminderEventDateStored && scope.ngModel.link && !scope.isStartDateEnabled) {
                            scope.allowedDateRange.min = null;
                        } else {
                            scope.allowedDateRange.min = formatter.getFormattedFrontendDate(new Date());
                        }
                    }
                }, true);

				scope.$watch('nextDueDate', function (newVal, oldVal) {
					if (newVal) {
						validateNextDueDate(newVal);

						// nextDueDate has changed values, so clear out alert array
						if (oldVal && oldVal !== newVal) {
							scope.ngModel.alert = [];
						}
					} else if (!newVal && oldVal) {
						scope.ngModel.alert = [];
					}
				}, true);

				scope.isAlertOptionValid = function (item) {
					if(typeof scope.ngModel === "undefined"){
						return false;
					}
					var currentDate = new Date(),
					    valid = false,
					    dayDiff, nextDue;

					if (scope.nextDueDate && scope.ngModel.notificationStatus === 'On') {
						nextDue = new Date(scope.nextDueDate);
						dayDiff = getDayDifference(currentDate, nextDue);
						valid = dayDiff > dateOptionMap[item];
					}

					return valid;
				};

				scope.toggleSelection = function (item) {
					var idx = scope.ngModel.alert.indexOf(item);
					if (idx > -1) {
						scope.ngModel.alert.splice(idx, 1);
					} else {
						scope.ngModel.alert.push(item);
					}
				};
			}
		};
	});
});
